package com.androidbook.pettracker3;

import android.app.AlertDialog;
import android.content.ContentUris;
import android.content.Context;
import android.content.DialogInterface;
import android.database.Cursor;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.androidbook.pettracker3.PetTrackerDatabase.PetType;
import com.androidbook.pettracker3.PetTrackerDatabase.Pets;

public class PetTrackerListActivity extends PetTrackerActivity {

	protected Cursor mCursor;
	
	@Override
	public void onCreate(Bundle savedInstanceState) {

		super.onCreate(savedInstanceState);
		setContentView(R.layout.showpets);

		// Wypełnienie kontrolki ListyView danymi z bazy.
		fillPetList();

		// Obsługa przycisku Dodaj zwierzaki
		final Button gotoEntry = (Button) findViewById(R.id.ButtonEnterMorePets);
		gotoEntry.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {

				// I to wszystko. Kończymy i wracamy do ekranu początkowego.
				finish();
			}
		});
	}

	public void fillPetList() {
	
		// Zapytanie SQL.
		SQLiteQueryBuilder queryBuilder = new SQLiteQueryBuilder();
		queryBuilder.setTables(Pets.PETS_TABLE_NAME + ", "
				+ PetType.PETTYPE_TABLE_NAME);
		queryBuilder.appendWhere(Pets.PETS_TABLE_NAME + "."
				+ Pets.PET_TYPE_ID + "=" + PetType.PETTYPE_TABLE_NAME + "."
				+ PetType._ID);

		// Pobranie obiektu bazy danych i wykonanie zapytania.
		String asColumnsToReturn[] = {
				Pets.PETS_TABLE_NAME + "." + Pets.PET_NAME,
				Pets.PETS_TABLE_NAME + "." + Pets.PET_IMAGE_URI,
				Pets.PETS_TABLE_NAME + "." + Pets._ID,
				Pets.PETS_TABLE_NAME + "." + Pets.PET_IMAGE_ID,
				PetType.PETTYPE_TABLE_NAME + "." + PetType.PET_TYPE_NAME };
		
		mCursor = queryBuilder.query(mDB, asColumnsToReturn, null, null,
				null, null, Pets.DEFAULT_SORT_ORDER);
		startManagingCursor(mCursor);
		
        // Użycie adaptera do powiązania danych z kontrolką ListView, w której każdy element jest prezentowany
        // przy wykorzystaniu układu pet_item (pet_item.xml).       
		PetListAdapter adapter = new PetListAdapter(this, mCursor);	
		ListView av = (ListView) findViewById(R.id.petList);
		av.setAdapter(adapter);

		// Odbieranie i obsługa kliknięć elementów kontrolki ListView
		av.setOnItemClickListener(new AdapterView.OnItemClickListener() {
			public void onItemClick(AdapterView<?> parent, View view,
					int position, long id) {
				
				final long deletePetId = id;

				// Używamy okna dialogowego AlertDialog, by potwierdzić chęć usunięcia zwierzaka.
				new AlertDialog.Builder(PetTrackerListActivity.this).setMessage(
						"Czy usunąć rekord zwierzaka?").setPositiveButton("Delete",
						new DialogInterface.OnClickListener() {

							public void onClick(DialogInterface dialog,
									int which) {

								// Usuwamy zwierzaka.
								deletePet(deletePetId);
								// Odświeżenie danych w kursorze, a zatem także w kontrolce listy.
								refillPetList(); 
							}
						}).show();
			}
		});
	}
	
	// W razie konieczności odświeżamy adapter ListAdapter.
	public void refillPetList() {
		mCursor.requery();
		PetListAdapter adapter = new PetListAdapter(this, mCursor);
		ListView av = (ListView) findViewById(R.id.petList);
		av.setAdapter(adapter);
	}

	// Usuwamy zwierzaka na podstawie identyfikatora.
	public void deletePet(Long id) {
		String astrArgs[] = { id.toString() };
		mDB.delete(Pets.PETS_TABLE_NAME, Pets._ID + "=?", astrArgs);
	}

	// Klasa zawierająca obiekty reprezentujące elementy kontrolki ListView.
	static class PetListItemContainer  {
		
	    TextView mPetName; 
	    TextView mPetType; 
	    ImageView mPetPic; 
	}

	// Oto niestandardowy adapter, zawierający wymagane kolumny.
	// Odwzorowuje on obiekt Cursor na tablice obiektów PetRecor.
	public class PetListAdapter extends BaseAdapter {

		private PetRecord[] mPets;
		private Context mContext;
		private LayoutInflater mInflater;

		public PetListAdapter(Context context, Cursor curs) {
			mContext = context;
			mInflater = LayoutInflater.from(mContext); 

			int iNumPets = curs.getCount();
			mPets = new PetRecord[iNumPets];

			// Pobranie zawartości kursora.
			curs.moveToFirst();
			for (int i = 0; i < iNumPets; i++) {
				final String strName = curs.getString(curs.getColumnIndex(Pets.PET_NAME));
				final String strType = curs.getString(curs.getColumnIndex(PetType.PET_TYPE_NAME));
				final String strImageUriPath = curs.getString(curs.getColumnIndex(Pets.PET_IMAGE_URI));	
				final long petId = curs.getLong(curs.getColumnIndex(Pets._ID));
				final long petImageId = curs.getLong(curs.getColumnIndex(Pets.PET_IMAGE_ID));

				mPets[i] = new PetRecord(strName, strType, strImageUriPath, petImageId, petId);
				curs.moveToNext();
			}

		}
		
		public int getCount() {
			return mPets.length;
		}

		public Object getItem(int position) {
			return mPets[position];
		}

		public long getItemId(int position) {
			return mPets[position].getPetId();
		}
	
		// Metoda zwraca obiekt View odpowiadający poszczególnym elementom podrzędnym kontrolki ListView.
		// Każdy element ListView jest tworzony na podstawie układu pet_item, w którym zostają zapisane odpowiednie dane.
		// Taki układ można by także wygenerować programowo (patrz komentarz poniżej).
		// Wskazówka: Skorzystaliśmy z właściwości Tag obiektu View, by dodać do niego metadane o zwierzaku, który
		// reprezentuje dany element listy.
		public View getView(int position, View convertView, ViewGroup parent) {

			PetListItemContainer i;
			
			if(convertView == null)
			{
				// Tworzymy nowy element listy.
				convertView = (RelativeLayout)mInflater.inflate(R.layout.pet_item, null); 
				i = new PetListItemContainer();
				i.mPetName = (TextView) convertView.findViewById(R.id.TextView_PetName);
				i.mPetType = (TextView) convertView.findViewById(R.id.TextView_PetType);
				i.mPetPic = (ImageView) convertView.findViewById(R.id.ImageView_PetPic);
				convertView.setTag(i);
				
			} 
			else
			{
				// Pobranie elementu ze znacznika. 
				i = (PetListItemContainer) convertView.getTag();
			}
						
			i.mPetName.setText(mPets[position].getPetName());
			i.mPetType.setText(mPets[position].getPetType());
	
			// Obrazek zwierzaka.
			// Info: Solidniejsza implementacja powinna także sprawdzać, czy obrazek wciąż jest dostępny na karcie SD. 
			// Tutaj chodziło nam jednak o zachowanie prostoty przykładu. 
	        if(mPets[position].getPetImageId() != PetRecord.INVALID_PET_IMAGE_ID)
	        {
	        	Uri baseUri = Uri.parse(mPets[position].getPetImageUriPath());
			    Uri imageUri = ContentUris.withAppendedId(baseUri, mPets[position].getPetImageId());
		        i.mPetPic.setImageURI(imageUri); 
	        }  
		  		
			return convertView;
			
			// Info - Programowy sposób tworzenia układu na bazie pliku XML 
			// (Dodatkowe informacje w pliku PetListItemView.java)
			
			// PetListItemView i = new PetListItemView(mContext, mPets[position]);
			//	return i;
		}	
	}
	
}

